<?php
/**
 *  Module made by Nukium
 *
 *  @author    Nukium
 *  @copyright 2021 Nukium SAS
 *  @license   All rights reserved
 *
 * ███    ██ ██    ██ ██   ██ ██ ██    ██ ███    ███
 * ████   ██ ██    ██ ██  ██  ██ ██    ██ ████  ████
 * ██ ██  ██ ██    ██ █████   ██ ██    ██ ██ ████ ██
 * ██  ██ ██ ██    ██ ██  ██  ██ ██    ██ ██  ██  ██
 * ██   ████  ██████  ██   ██ ██  ██████  ██      ██
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

use PrestaShop\PrestaShop\Core\Module\WidgetInterface;

if (!class_exists('NkmCsv')) {
    require_once(dirname(__FILE__).'/lib/NkmCsv.php');
}

class Nkmfeaturemanager extends Module implements WidgetInterface
{
    protected $config_form = false;

    private $default_multiple_value_separator = '|';
    private $default_field_separator = ';';
    private $imageType = 'jpg';

    public static $lang_doc = array('fr', 'en', 'es');

    public function __construct()
    {
        $this->name = 'nkmfeaturemanager';
        $this->tab = 'export';
        $this->version = '1.1.3';
        $this->author = 'Nukium';
        $this->module_key = '48cbc11e5808e6b860608068688f1b68';
        $this->need_instance = 0;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Advanced features: Import / Export + Images');
        $this->description = $this->l('An easy way to manage products features using Import / Export. Enhance features values with images on product pages.');

        $this->ps_versions_compliancy = array(
            'min' => '1.7',
            'max' => _PS_VERSION_
        );

        if (!defined('NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR')) {
            define('NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR', _PS_IMG_DIR_.'f'.DIRECTORY_SEPARATOR);
        }
        if (!defined('NKMFEATUREMANAGER_FEATURE_VALUE_IMG_URI')) {
            define('NKMFEATUREMANAGER_FEATURE_VALUE_IMG_URI', _PS_IMG_.'f'.DIRECTORY_SEPARATOR);
        }
    }

    public function install()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        Configuration::updateValue('NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR', $this->default_multiple_value_separator);
        Configuration::updateValue('NKMFEATUREMANAGER_FIELD_SEPARATOR', $this->default_field_separator);
        Configuration::updateValue('NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR', $this->default_multiple_value_separator);
        Configuration::updateValue('NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR', $this->default_field_separator);
        Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME', 0);
        Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE', 0);
        Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_POSITION', 0);

        if (!is_dir(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR)) {
            mkdir(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR, 0775);
        }

        return parent::install() &&
            $this->registerHook('displayBackOfficeHeader')
            && $this->registerHook('displayHeader')
            && $this->registerHook('displayFeatureValueForm')
            && $this->registerHook('actionFeatureValueDelete')
            && $this->registerHook('displayFeatureValuePostProcess')
            && $this->registerHook('actionFeatureValueSave')
            && $this->registerHook('actionObjectFeatureValueAddAfter')
            && $this->registerHook('actionObjectFeatureValueUpdateAfter')
            && $this->registerHook('actionObjectFeatureValueDeleteBefore')
            && $this->registerHook('displayFooterProduct')
            && $this->registerHook('actionFrontControllerSetMedia');
    }

    public function uninstall()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        Configuration::deleteByName('NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR');
        Configuration::deleteByName('NKMFEATUREMANAGER_FIELD_SEPARATOR');
        Configuration::deleteByName('NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR');
        Configuration::deleteByName('NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR');
        Configuration::deleteByName('NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME');
        Configuration::deleteByName('NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE');
        Configuration::deleteByName('NKMFEATUREMANAGER_IMPORT_POSITION');

        return parent::uninstall();
    }


    public function getContent()
    {
        $this->context->smarty->assign('module_dir', $this->_path);

        if (in_array($this->context->language->iso_code, self::$lang_doc)) {
            $this->context->smarty->assign('iso_lang', $this->context->language->iso_code);
        } else {
            $this->context->smarty->assign('iso_lang', 'en');
        }

        $output = '';


        if (((bool)Tools::isSubmit('submitNkmfeaturemanagerModule')) == true) {
            $output = $this->postProcess();
        }

        $this->context->smarty->assign('module_dir', $this->_path);

        $footer = $this->context->smarty->fetch($this->local_path.'views/templates/admin/configure.tpl');

        return $output.$this->renderForm().$footer;
    }


    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitNkmfeaturemanagerModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), 
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm($this->getConfigForm());
    }


    protected function getConfigForm()
    {
        $fields_form = array();

        $fields_form[0] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Export of products features'),
                    'icon' => 'icon-file-excel-o'
                ),
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Language of the file'),
                        'name' => 'NKMFEATUREMANAGER_EXPORT_LANGUAGE',
                        'required' => true,
                        'options' => array(
                            'query' => Language::getLanguages(false),
                            'id' => 'id_lang',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Field separator'),
                        'name' => 'NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR',
                        'required' => true,
                        'class' => 'input fixed-width-sm',
                        'lang' => false,
                        'desc' => $this->l('e.g. Wool; Green; Organic'),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Multiple value separator'),
                        'name' => 'NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR',
                        'required' => true,
                        'class' => 'input fixed-width-sm',
                        'lang' => false,
                        'desc' => $this->l('e.g. Wool; Red| Blue| Green; Organic'),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Export'),
                    'class' => 'btn btn-default',
                    'id' => 'submitNkmfeaturemanagerModuleExport',
                    'name' => 'submitNkmfeaturemanagerModuleExport',
                    'icon' => 'process-icon-upload'
                )
            )
        );

        $fields_form[1] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Import of products features'),
                    'icon' => 'icon-file-excel-o'
                ),
                'description' => $this->l('The import file format must match the file exported before.'),
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Language of the file'),
                        'name' => 'NKMFEATUREMANAGER_IMPORT_LANGUAGE',
                        'required' => true,
                        'options' => array(
                            'query' => Language::getLanguages(false),
                            'id' => 'id_lang',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Field separator'),
                        'name' => 'NKMFEATUREMANAGER_FIELD_SEPARATOR',
                        'required' => true,
                        'class' => 'input fixed-width-sm',
                        'lang' => false,
                        'desc' => $this->l('e.g. Wool; Green; Organic'),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Multiple value separator'),
                        'name' => 'NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR',
                        'required' => true,
                        'class' => 'input fixed-width-sm',
                        'lang' => false,
                        'desc' => $this->l('e.g. Wool; Red| Blue| Green; Organic'),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Update product name on import'),
                        'name' => 'NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME',
                        'required' => true,
                        'is_bool' => true,
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Yes')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('No')
                            )
                        ),
                        'lang' => false
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Delete all products features before import'),
                        'name' => 'NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE',
                        'required' => true,
                        'is_bool' => true,
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Yes')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('No')
                            )
                        ),
                        'lang' => false
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Update features positions on import'),
                        'name' => 'NKMFEATUREMANAGER_IMPORT_POSITION',
                        'required' => true,
                        'is_bool' => true,
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Yes')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('No')
                            )
                        ),
                        'lang' => false
                    ),
                    array(
                        'type' => 'file',
                        'label' => $this->l('File'),
                        'name' => 'file',
                        'required' => true,
                        'desc' => $this->l('File extension: .csv (only UTF-8 encoding is allowed)'),
                    )
                ),
                'submit' => array(
                    'title' => $this->l('Import'),
                    'class' => 'btn btn-default',
                    'id' => 'submitNkmfeaturemanagerModuleImport',
                    'name' => 'submitNkmfeaturemanagerModuleImport',
                    'icon' => 'process-icon-download'
                )
            )
        );

        return $fields_form;
    }


    protected function getConfigFormValues()
    {
        return array(
            'NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME' => Tools::getValue(
                'NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME',
                Configuration::get('NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME')
            ),
            'NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE' => Tools::getValue(
                'NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE',
                Configuration::get('NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE')
            ),
            'NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR' => Tools::getValue(
                'NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR',
                Configuration::get('NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR')
            ),
            'NKMFEATUREMANAGER_FIELD_SEPARATOR' => Tools::getValue(
                'NKMFEATUREMANAGER_FIELD_SEPARATOR',
                Configuration::get('NKMFEATUREMANAGER_FIELD_SEPARATOR')
            ),
            'NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR' => Tools::getValue(
                'NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR',
                Configuration::get('NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR')
            ),
            'NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR' => Tools::getValue(
                'NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR',
                Configuration::get('NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR')
            ),
            'NKMFEATUREMANAGER_EXPORT_LANGUAGE' => Tools::getValue(
                'NKMFEATUREMANAGER_EXPORT_LANGUAGE',
                $this->context->language->id
            ),
            'NKMFEATUREMANAGER_IMPORT_LANGUAGE' => Tools::getValue(
                'NKMFEATUREMANAGER_IMPORT_LANGUAGE',
                $this->context->language->id
            ),
            'NKMFEATUREMANAGER_IMPORT_POSITION' => Tools::getValue(
                'NKMFEATUREMANAGER_IMPORT_POSITION',
                Configuration::get('NKMFEATUREMANAGER_IMPORT_POSITION')
            ),
        );
    }


    protected function postProcess()
    {
        $output = null;

        $form_values = $this->getConfigFormValues();

        $moduleSeoSa = false;
        if (Module::isInstalled('seosaextendedfeatures') && Module::isEnabled('seosaextendedfeatures')) {
            $moduleSeoSa = true;
        }

        if (Tools::getValue('submitNkmfeaturemanagerModuleExport')) {
            if (empty($form_values['NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR'])) {
                $output .= $this->displayError($this->l('Multiple value separator is required and can\'t be empty.'));
            }
            if (empty($form_values['NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR'])) {
                $output .= $this->displayError($this->l('Field separator is required and can\'t be empty.'));
            }

            if (is_null($output)) {
                Configuration::updateValue('NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR', $form_values['NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR']);
                Configuration::updateValue('NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR', $form_values['NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR']);

                $id_lang_selected = $form_values['NKMFEATUREMANAGER_EXPORT_LANGUAGE'];

                if (ob_get_level() && ob_get_length() > 0) {
                    ob_clean();
                }

                $csv_file = new NkmCsv();
                $csv_file->csvDelimeter = $form_values['NKMFEATUREMANAGER_EXPORT_FIELD_SEPARATOR'];
                $csv_header = array('ID', 'PDT_REF', 'PDT_NAME', 'PDT_MANUFACTURER');

                try {
                    $cache_features = Feature::getFeatures($id_lang_selected);
                    foreach ($cache_features as $value) {
                        array_push($csv_header, $value['name']);
                    }
                } catch (Exception $e) {
                    $output .= $this->displayError($e->getMessage());
                }

                $csv_file->createTemplate($csv_header);

                try {
                    foreach (Product::getProducts($this->context->language->id, 0, 0, 'id_product', 'DESC') as $value) {
                        $csv_line = array();

                        if ($moduleSeoSa) {
                            $productFeaturesStatic = self::getFeaturesStaticSEOSA($value['id_product']);
                        } else {
                            $productFeaturesStatic = ProductCore::getFeaturesStatic($value['id_product']);
                        }

                        $product_features = array();
                        foreach ($productFeaturesStatic as $v) {
                            if (isset($product_features[$v['id_feature']])) {
                                $product_features[$v['id_feature']] .= ':'.$v['id_feature_value'];
                            } else {
                                $product_features[$v['id_feature']] = $v['id_feature_value'];
                            }
                        }

                        $csv_line[] = $value['id_product'];
                        $csv_line[] = $value['reference'];
                        $csv_line[] = $value['name'];
                        $csv_line[] = $value['manufacturer_name'];

                        foreach ($cache_features as $f) {
                            if (array_key_exists($f['id_feature'], $product_features)) {
                                $feature_values = '';
                                $ids_feature_value = explode(':', $product_features[$f['id_feature']]);

                                foreach ($ids_feature_value as $id) {
                                    $feature_value = new FeatureValue($id, $id_lang_selected);
                                    if (!empty($feature_values)) {
                                        $feature_values .= $form_values['NKMFEATUREMANAGER_EXPORT_MULTIPLE_VALUE_SEPARATOR'];
                                    }
                                    $feature_values .= $feature_value->value;
                                }
                                $csv_line[] = $feature_values;
                            } else {
                                $csv_line[] = '';
                            }
                        }
                        $csv_file->addEntry($csv_line);
                    }
                } catch (Exception $e) {
                    $output .= $this->displayError($e->getMessage());
                }

                if (is_null($output)) {
                    header('Content-type: text/csv');
                    header('Content-Type: application/force-download; charset=UTF-8');
                    header('Cache-Control: no-store, no-cache');
                    header('Content-disposition: attachment; filename="product_feature_export_'.date('Y-m-d_His').'.csv"');

                    exit($csv_file->buildDoc());
                }
            }
        }


        if (Tools::getValue('submitNkmfeaturemanagerModuleImport')) {
            if (empty($form_values['NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR'])) {
                $output .= $this->displayError($this->l('Multiple value separator is required and can\'t be empty.'));
            }
            if (empty($form_values['NKMFEATUREMANAGER_FIELD_SEPARATOR'])) {
                $output .= $this->displayError($this->l('Field separator is required and can\'t be empty.'));
            }

            if (is_null($output)) {
                Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME', $form_values['NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME']);
                Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE', $form_values['NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE']);
                Configuration::updateValue('NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR', $form_values['NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR']);
                Configuration::updateValue('NKMFEATUREMANAGER_FIELD_SEPARATOR', $form_values['NKMFEATUREMANAGER_FIELD_SEPARATOR']);
                Configuration::updateValue('NKMFEATUREMANAGER_IMPORT_POSITION', $form_values['NKMFEATUREMANAGER_IMPORT_POSITION']);

                $id_lang_selected = $form_values['NKMFEATUREMANAGER_IMPORT_LANGUAGE'];

                $input_file = $this->getCsvFile();
                if (is_array($input_file)) {
                    $csv_file = new NkmCSVReader();
                    $csv_file->separator = $form_values['NKMFEATUREMANAGER_FIELD_SEPARATOR'];
                    $csv_content = $csv_file->parse_file($input_file['file_path'], true, false);

                    if (!$csv_content) {
                        $output .= $this->displayWarning($this->l('File content empty.'));
                    } else {
                        $featuresCache = array();
                        $features = Feature::getFeatures($id_lang_selected);
                        if (is_array($features) && count($features) > 0) {
                            foreach ($features as $v) {
                                $featuresCache[$v['id_feature']] = $v['name'];
                            }
                        }
                        $featuresValuesCache = array();


                        foreach ($csv_content as $line) {
                            try {
                                $id_product = $line['ID'];
                                $product = new Product($id_product);

                                if ($product) {
                                    $features = array();
                                    foreach ($line as $key => $value) {
                                        if ($key != 'ID' &&
                                            $key != 'PDT_REF' &&
                                            $key != 'PDT_NAME' &&
                                            $key != 'PDT_MANUFACTURER'
                                        ) {
                                            $features[(string)trim($key)] = trim($value);
                                        }
                                    }

                                    if ((int)$form_values['NKMFEATUREMANAGER_IMPORT_DELETE_PRODUCT_FEATURE'] == 1) {
                                        $product->deleteProductFeatures();
                                    }
                                    $i = 0;
                                    foreach ($features as $feature_name => $feature_value) {
                                        if (empty($feature_name)) {
                                            continue;
                                        }

                                        if (in_array($feature_name, $featuresCache)) {
                                            $id_feature = array_search($feature_name, $featuresCache);
                                        } else {
                                            $feature = new Feature();
                                            $feature->name = array_fill_keys(Language::getIDs(), (string)$feature_name);
                                            if ((int)Configuration::get('NKMFEATUREMANAGER_IMPORT_POSITION') == 1) {
                                                $feature->position = (int)$i;
                                                $i++;
                                            } else {
                                                $feature->position = Feature::getHigherPosition() + 1;
                                            }
                                            $feature->add();

                                            $id_feature = (int)$feature->id;

                                            $featuresCache[$id_feature] = $feature_name;
                                        }

                                        if (empty($feature_value)) {
                                            continue;
                                        }

                                        $tab_feature_values = explode($form_values['NKMFEATUREMANAGER_MULTIPLE_VALUE_SEPARATOR'], $feature_value);

                                        $position = null;
                                        if ($moduleSeoSa) {
                                            $position = 1;
                                        }
                                        foreach ($tab_feature_values as $v) {
                                            if (in_array($id_feature . '-' . $v, $featuresValuesCache)) {
                                                $id_feature_value = array_search($id_feature . '-' . $v, $featuresValuesCache);
                                            } else {
                                                $id_feature_value = (int)FeatureValue::addFeatureValueImport(
                                                    $id_feature,
                                                    $v,
                                                    null,
                                                    $id_lang_selected,
                                                    false
                                                );
                                                $featuresValuesCache[$id_feature_value] = $id_feature . '-' . $v;
                                            }

                                            self::addFeatureValueToProduct($product->id, $id_feature, $id_feature_value, $position);
                                            if ($moduleSeoSa) {
                                                $position++;
                                            }
                                        }
                                    }

                                    $line['PDT_NAME'] = trim($line['PDT_NAME']);
                                    if (!empty($line['PDT_NAME']) && (int)$form_values['NKMFEATUREMANAGER_IMPORT_UPDATE_PRODUCT_NAME'] == 1) {
                                        $product->name[$id_lang_selected] = $line['PDT_NAME'];
                                        $product->update();
                                    }
                                } else {
                                    $output .= $this->displayWarning(sprintf($this->l('Product ID #%d doesn\'t exist.'), $id_product));
                                }
                            } catch (Exception $e) {
                                $output .= $this->displayError($e->getMessage());
                            }
                        }

                        Feature::cleanPositions();

                        unlink($input_file['file_path']);
                    }
                } else {
                    $output .= $input_file;
                }
            }

            if (!$output) {
                $output = $this->displayConfirmation($this->l('Import successful'));
            }
        }
        return $output;
    }


    public static function addFeatureValueToProduct($id_product, $id_feature, $id_feature_value, $position = null)
    {
        $id_product = (int)$id_product;
        $id_feature = (int)$id_feature;
        $id_feature_value = (int)$id_feature_value;

        $data = array(
            'id_product' => $id_product,
            'id_feature' => $id_feature,
            'id_feature_value' => $id_feature_value
        );

        if (!is_null($position)) {
            $data['position'] = (int)$position;
        }

        $query = new DbQuery();
        $query->select('fp.*')
            ->from('feature_product', 'fp')
            ->where('fp.`id_feature` = ' . $id_feature)
            ->where('fp.`id_product` = ' . $id_product)
            ->where('fp.`id_feature_value` = ' . $id_feature_value);

        if (Db::getInstance()->getRow($query) && !is_null($position)) {
            return Db::getInstance()->update(
                'feature_product',
                array('position' => $position),
                ('`id_feature` = '.$id_feature.' AND `id_product` = '.$id_product.' AND `id_feature_value` = '.$id_feature_value)
            );
        } else {
            return Db::getInstance()->insert('feature_product', $data, false, true, Db::INSERT_IGNORE);
        }
    }

    public static function getFeaturesStaticSEOSA($id_product)
    {
        return Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS(
            'SELECT fp.id_feature, fp.id_product, fp.id_feature_value, fp.position
            FROM `' . _DB_PREFIX_ . 'feature_product` fp
            LEFT JOIN `' . _DB_PREFIX_ . 'feature_value` fv ON fp.id_feature_value = fv.id_feature_value
            WHERE `id_product` = ' . (int)$id_product . '
            ORDER BY fp.id_product, fp.id_feature, fp.position ASC'
        );
    }



    public function hookDisplayBackOfficeHeader()
    {
        if (Tools::getValue('module_name') == $this->name
            || (isset($this->context->controller) && get_class($this->context->controller) == 'AdminFeaturesController')
            ) {
            if (method_exists($this->context->controller, 'addJquery')) {
                $this->context->controller->addJquery();
            }
            $this->context->controller->addJS($this->_path.'views/js/back.js');
        }
    }


    public function getCsvFile()
    {
        $result = array();

        if (isset($_FILES['file']) && is_uploaded_file($_FILES['file']['tmp_name'])) {
            do {
                $uniqid = sha1(microtime());
            } while (file_exists(_PS_DOWNLOAD_DIR_.$uniqid));

            if (!move_uploaded_file($_FILES['file']['tmp_name'], _PS_DOWNLOAD_DIR_.$uniqid)) {
                return $this->displayError($this->l('Failed to copy the file.'));
            }

            $_POST['file_name'] = $_FILES['file']['name'];
            @unlink($_FILES['file']['tmp_name']);

            $result['file'] = $uniqid;
            $result['file_path'] = _PS_DOWNLOAD_DIR_.$uniqid;
            $result['mime'] = $_FILES['file']['type'];
        } elseif (array_key_exists('file', $_FILES) && (int)$_FILES['file']['error'] === 1) {
            $max_upload = (int)ini_get('upload_max_filesize');
            $max_post = (int)ini_get('post_max_size');
            $upload_mb = min($max_upload, $max_post);
            $result = $this->displayError(sprintf(
                $this->l('The file %1$s exceeds the size allowed by the server. The limit is set to %2$d MB.'),
                '<b>'.$_FILES['file']['name'].'</b> ',
                '<b>'.$upload_mb.'</b>'
            ));
        } else {
            $result = $this->displayError($this->l('File missing.'));
        }

        return $result;
    }



    public function hookActionFeatureValueDelete($params)
    {
    }


    public function hookActionFeatureValueSave($params)
    {
        if (!$params['id_feature_value']) {
            return;
        }

        if (isset($_FILES['feature_value_img'])
            && isset($_FILES['feature_value_img']['tmp_name'])
            && !empty($_FILES['feature_value_img']['tmp_name'])
            && (int)$params['id_feature_value'] > 0) {
            $file_name = (int)$params['id_feature_value'].'.'.$this->imageType;
            if (!ImageManager::resize($_FILES['feature_value_img']['tmp_name'], NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$file_name)) {
                return $this->displayError($this->l('An error occurred while attempting to upload the file.'));
            }
        }
    }


    public function hookDisplayFeatureValuePostProcess($params)
    {
        if (isset($_FILES['feature_value_img'])
            && isset($_FILES['feature_value_img']['tmp_name'])
            && !empty($_FILES['feature_value_img']['tmp_name'])) {
            if ($error = ImageManager::validateUpload($_FILES['feature_value_img'], 4000000)) {
                return $error;
            }
        }
    }


    public function hookDisplayFeatureValueForm()
    {
        if (Tools::getValue('id_feature_value') !== false && (int)Tools::getValue('id_feature_value') > 0) {
            if (file_exists(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.(int)Tools::getValue('id_feature_value').'.'.$this->imageType)) {
                $image = NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.(int)Tools::getValue('id_feature_value').'.'.$this->imageType;
                $image_url = ImageManager::thumbnail($image, 'feature_value_'.(int)Tools::getValue('id_feature_value').'.'.$this->imageType, 100, $this->imageType, true, true);
                $image_size = file_exists($image) ? filesize($image) / 1000 : false;
                $delete_url = '#';
            }
        }

        $uploader = new HelperUploader();
        $uploader->setName('feature_value_img');
        $uploader->setFiles(array(
            0 => array(
                'type'       => HelperUploader::TYPE_IMAGE,
                'image'      => isset($image_url)?$image_url:null,
                'size'       => isset($image_size)?$image_size:null,
                'delete_url' => isset($image_url)?$delete_url:null
            )));

        $this->context->smarty->assign(array(
            'feature_value_img' => $uploader->render()
        ));

        return $this->display(__FILE__, 'feature_value_form.tpl');
    }

    public function hookActionObjectFeatureValueAddAfter($params, $force_delete = false)
    {
        if (Validate::isLoadedObject($params['object']) && (Tools::getValue('feature_value_img_delete') || $force_delete)) {
            if (file_exists(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$params['object']->id.'.'.$this->imageType)) {
                unlink(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$params['object']->id.'.'.$this->imageType);
            }

            $params['object']->deleteImage(true);

            $types = ImageType::getImagesTypes();
            foreach ($types as $image_type) {
                if (file_exists(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$params['object']->id.'-'.Tools::stripslashes($image_type['name']).'.'.$this->imageType)) {
                    unlink(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$params['object']->id.'-'.Tools::stripslashes($image_type['name']).'.'.$this->imageType);
                }
            }
        }
    }

    public function hookActionObjectFeatureValueUpdateAfter($params)
    {
        $this->hookActionObjectFeatureValueAddAfter($params);
    }

    public function hookActionObjectFeatureValueDeleteBefore($params)
    {
        $this->hookActionObjectFeatureValueAddAfter($params, true);
    }

    public function renderWidget($hookName, array $params)
    {
        switch ($hookName) {
            case 'displayFooterProduct':
                $this->smarty->assign($this->getWidgetVariables($hookName, $params));
                return $this->fetch('module:'.$this->name.'/views/templates/hook/feature_value_img.tpl');

            default:
                break;
        }
    }

    public function getWidgetVariables($hookName, array $params)
    {
        switch ($hookName) {
            case 'displayFooterProduct':
                $feature_value_img = array();
                if (isset($params['product']['features'])
                    && is_array($params['product']['features'])
                    && count($params['product']['features']) > 0) {
                    $features = Product::getFeaturesStatic($params['product']['id']);
                    if (is_array($features) && count($features) > 0) {
                        foreach ($features as $value) {
                            if (file_exists(NKMFEATUREMANAGER_FEATURE_VALUE_IMG_DIR.$value['id_feature_value'].'.'.$this->imageType)) {
                                $feature_value_lang = FeatureValue::getFeatureValueLang($value['id_feature_value']);
                                $feature = Feature::getFeature($this->context->language->id, $value['id_feature']);
                                $feature_value_img[$value['id_feature_value']] = array(
                                    'src' => NKMFEATUREMANAGER_FEATURE_VALUE_IMG_URI.$value['id_feature_value'].'.'.$this->imageType,
                                    'alt' => FeatureValue::selectLang($feature_value_lang, $this->context->language->id),
                                    'feature_name' => $feature['name']);
                            }
                        }
                    }
                }
                return array('feature_value_img' => $feature_value_img);

            default:
                break;
        }
        return array();
    }

    public function hookActionFrontControllerSetMedia($params)
    {
        if (isset($this->context->controller) && get_class($this->context->controller) == 'ProductController') {
            $this->context->controller->registerJavascript(
                'module-nkmfeaturemanager-frontjs',
                'modules/'.$this->name.'/views/js/front.js',
                array('position' => 'bottom', 'priority' => 100)
            );
            $this->context->controller->registerStylesheet(
                'module-nkmfeaturemanager-frontcss',
                'modules/'.$this->name.'/views/css/front.css',
                array('media' => 'all')
            );
        }
    }
}
