<?php
/**
* 2007-2018 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author PrestaShop SA <contact@prestashop.com>
*  @copyright  2007-2022 PrestaShop SA
*  @license    http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')){
    exit;
}

class OrangeMoneyCMR extends PaymentModule
{
	/**
	 * Attributes
	 */
    private $_html = '';
	private $_postErrors = array();
	
    public $production_mode;
    public $authHeader;
    public $accessToken;
    public $merchantKey;
	public $tokenUrl;
	public $paymentUrl;
	public $statusUrl;
	public $merchantName;
	public $currency;
	public $language;
	public $base_link;

	/**
	 * Constructor
	 */
	public function __construct()
	{
        $this->name = 'orangemoneycmr';
        $this->tab = 'payments_gateways';
        $this->version = '4.0.0';
        $this->author = 'AByster Consulting.';
		$this->module_key ='63dca22c036713e9e6b47af3a64f4f4c';
        $this->controllers = array('validation', 'finalisation');
		$this->currencies = true;
		$this->language = 'fr';
		$this->base_link = '';
        $this->currencies_mode = 'checkbox';
        $this->bootstrap = true;
        parent::__construct();
        $this->page = basename(__FILE__, '.php');
        $this->displayName = $this->l('Payment by Orange Money CMR');
        $this->description = $this->l('Accepts payments by Orange Money Cameroon.');
        $this->confirmUninstall = $this->l('Are you sure you want to delete your details ?');

		/* Loading module's configurations */
        $this->production_mode = Configuration::get('AB_PRODUCTION_MODE');
        $this->authHeader = Configuration::get('AB_AUTH_HEADER');
        $this->accessToken = Configuration::get('AB_ACCESS_TOKEN');
        $this->merchantKey = Configuration::get('AB_MERCHANT_KEY');
		$this->merchantName = Configuration::get('AB_MERCHANT_NAME');
		$this->tokenUrl = Configuration::get('AB_TOKEN_URL');
        $this->paymentUrl = Configuration::get('AB_PAYMENT_URL');
		$this->statusUrl = Configuration::get('AB_STATUS_URL');
		$this->currency = Configuration::get('AB_CURRENCY');
		
        if (!count(Currency::checkPaymentCurrencies($this->id))) {
			$this->warning = $this->l('No currency has been set for this module.');
        }
	}

	/**
	 * Install module
	 */
	public function install()
	{
        $this->writeInlogFile($this->l('Installing module... '), 'INFO');
		$this->writeInlogFile($this->l('Installing hooks...'), 'INFO');
		
		/* install hooks */
        if (!parent::install() || !$this->registerHook('paymentOptions') || !$this->registerHook('paymentReturn')) {
			return false;
        }
        $this->writeInlogFile($this->l('Hooks installed !!!'), 'INFO');

        /* install Tables */
        if (!$this->createTables()) {
			return false;
        }
        $this->writeInlogFile($this->l('Creating configuration options ...'), 'INFO');

        /* Initialise module's configurations */
		Configuration::updateValue('AB_AUTH_HEADER', '');
		Configuration::updateValue('AB_ACCESS_TOKEN', '');
        Configuration::updateValue('AB_MERCHANT_KEY', '');
		Configuration::updateValue('AB_MERCHANT_NAME', '');
		Configuration::updateValue('AB_PRODUCTION_MODE', 0);
		Configuration::updateValue('AB_CURRENCY', 'OUV');
		Configuration::updateValue('AB_TOKEN_URL', 'https://api.orange.com/oauth/v3/token');
        Configuration::updateValue('AB_PAYMENT_URL', 'https://api.orange.com/orange-money-webpay/dev/v1/webpayment');
        Configuration::updateValue('AB_STATUS_URL', 'https://api.orange.com/orange-money-webpay/dev/v1/transactionstatus');
        $this->writeInlogFile($this->l('Configuration options created ...'), 'INFO');

        $this->writeInlogFile($this->l('Module installed sucessfully !!!'), 'INFO');
        return true;
	}

	/**
	 * Creates module's tables
	 */
	public function createTables(){

        $sql = array();
        $sql[] = "CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "orangemoneycmr_payments` (
            `id_payment` INT(11) NOT NULL PRIMARY KEY AUTO_INCREMENT,
            `ref_order` VARCHAR(255),
            `customer_name` VARCHAR(255),
            `customer_email` VARCHAR(255),
            `customer_tel` FLOAT,
            `request_status` INT(11),
            `payment_amount` INT(11),
            `payment_status` VARCHAR(255),
            `message` VARCHAR(255),
            `pay_token` VARCHAR(255),
            `payment_url` VARCHAR(255),
            `notif_token` VARCHAR(255),
            `transaction_id` VARCHAR(255),
            `date_add` TIMESTAMP  DEFAULT CURRENT_TIMESTAMP
        ) ENGINE = " . _MYSQL_ENGINE_ ;

        $this->writeInlogFile($this->l('Creating module\'s tables... '), 'INFO');
        foreach ($sql as $q) {
			if (!DB::getInstance()->execute($q)) {
				return false;
			}
        }
        $this->writeInlogFile($this->l('Module\'s tables created !!!'), 'INFO');
        return true;
	}

	
	/**
	 * Uninstall the module
	 */
	public function uninstall()
	{
        $this->writeInlogFile($this->l('Uninstalling module... '), 'INFO');
        if (!parent::uninstall())
            return false;

        /* Delete module's tables */
        $this->writeInlogFile($this->l('Deleting module\'s tables ...'), 'INFO');
        $sql = array();
        $sql[] = "DROP TABLE IF EXISTS `"._DB_PREFIX_."orangemoneycmr_payments`";
        foreach ($sql as $q) {
			if (!DB::getInstance()->execute($q)) {
				return false;
			}
        }

        $this->writeInlogFile($this->l('Module\'s tables deleted !!!'), 'INFO');

        /* Clean configuration table */
        $this->writeInlogFile($this->l('Removing module\'s options from conffigurations table ...'), 'INFO');
        Configuration::deleteByName('AB_PRODUCTION_MODE');
        Configuration::deleteByName('AB_ACCESS_TOKEN');
        Configuration::deleteByName('AB_AUTH_HEADER');
        Configuration::deleteByName('AB_MERCHANT_KEY');
		Configuration::deleteByName('AB_MERCHANT_NAME');
		Configuration::deleteByName('AB_TOKEN_URL');
        Configuration::deleteByName('AB_PAYMENT_URL');
		Configuration::deleteByName('AB_STATUS_URL');
		Configuration::deleteByName('AB_CURRENCY');
        $this->writeInlogFile($this->l('Module\'s options removed from conffigurations table !!!'), 'INFO');

        $this->writeInlogFile($this->l('Module uninstalled sucessfully !!!'), 'INFO');
        return true;
	}

	/**
	 * Validates the settings form
	 */
	private function _postValidation()
	{
		if (Tools::isSubmit('btnSubmit'))
		{
			if (!Tools::getValue('merchant_name') ||
				!Tools::getValue('merchant_key') ||
				!Tools::getValue('auth_header') 
			   )
				$this->_postErrors[] = $this->l('please fill in the required fields');
		}
	}

	/**
	 * Process the form
	 */
	private function _postProcess()
	{
		if (Tools::isSubmit('btnSubmit'))
		{
			Configuration::updateValue('AB_MERCHANT_NAME', Tools::getValue('merchant_name'));
			Configuration::updateValue('AB_MERCHANT_KEY', Tools::getValue('merchant_key'));
			Configuration::updateValue('AB_AUTH_HEADER', Tools::getValue('auth_header'));

			if (Configuration::get('AB_PRODUCTION_MODE') != Tools::getValue('production_mode')) {
				Configuration::updateValue('AB_ACCESS_TOKEN', '');
				Configuration::updateValue('AB_PRODUCTION_MODE', Tools::getValue('production_mode'));
			}

			if (Configuration::get('AB_PRODUCTION_MODE')) {
				Configuration::updateValue('AB_CURRENCY', 'XAF');
				Configuration::updateValue('AB_PAYMENT_URL', 'https://api.orange.com/orange-money-webpay/cm/v1/webpayment');
				Configuration::updateValue('AB_STATUS_URL', 'https://api.orange.com/orange-money-webpay/cm/v1/transactionstatus');
			} else {
				Configuration::updateValue('AB_CURRENCY', 'OUV');
				Configuration::updateValue('AB_PAYMENT_URL', 'https://api.orange.com/orange-money-webpay/dev/v1/webpayment');
				Configuration::updateValue('AB_STATUS_URL', 'https://api.orange.com/orange-money-webpay/dev/v1/transactionstatus');
			}
		}
		$this->_html .= $this->displayConfirmation($this->l('Settings Successfully updated.'));
	}

	/**
	 * Display Orange money information templates
	 */
	private function _displayOrangeMoneyCMR()
	{
		$this->_html .= $this->display(__FILE__, 'infos.tpl');
	}

	/**
	 * Display configuration page
	 */
	public function getContent()
	{
		$this->_html = $this->l($this->displayName);
		if (Tools::isSubmit('btnSubmit'))
		{
			$this->_postValidation();
			if (!count($this->_postErrors)) {
				$this->_postProcess();
			} else {
				foreach ($this->_postErrors as $err) {
					$this->_html .= $this->displayError($err);	
				}
			}

		}
		$this->_displayOrangeMoneyCMR();
		$this->_html .= $this->renderForm();

		return $this->_html;
	}

    /**
	 * Hook payment return
	 */
	public function hookPaymentOptions($params)
	{
		if (!$this->active) {
            return;
        }
        if (!$this->checkCurrency($params['cart'])) {
            return;
		}
		
        $payment_options=array();
		$payment_options[] = $this->getExternalPaymentOption($params);
		
        return $payment_options;		
	}

	/**
	 * Get external payment option
	 */
	public function getExternalPaymentOption()
    {
        $externalOption = new PaymentOption();
        $externalOption->setCallToActionText($this->l('Payer par Orange Money CMR'))
					   ->setAction($this->context->link->getModuleLink($this->name, 'validation', array(), true))
					   ->setForm($this->generateForm());

        return $externalOption;
	}
	
	/**
	 * Generates payment form
	 */
	protected function generateForm()
    {
		$customer = new Customer($this->context->cart->id_customer);

        $this->context->smarty->assign(array(
            'action' => $this->context->link->getModuleLink($this->name, 'validation', array(), true),
            'nom' => $customer->firstname, 
			'email' => $customer->email
        ));
        return $this->context->smarty->fetch('module:orangemoneycmr/views/templates/front/payment_form.tpl');
    }

	/**
	 * Function to trigger on payment return
	 */
    public function hookPaymentReturn($params)
	{
		if (!$this->active)
			return;

		$smarty = $this->context->smarty;
		$state = $params['order']->current_state;
		$currency = new Currency($params['order']->id_currency);

		if ($state == Configuration::get('PS_OS_PAYMENT'))
		{
			$smarty->assign(array(
				'total_to_pay' => Tools::displayPrice($params['order']->total_paid, $currency, false),
				'status' => 'ok',
				'id_order' => $params['order']->reference
			));
			if (isset($params['order']->reference) && !empty($params['order']->reference))
				$smarty->assign('reference', $params['order']->reference);
		} 
		else {
			$smarty->assign(array(
				'status' => 'failed'
			));
		}
		return $this->context->smarty->fetch('module:orangemoneycmr/views/templates/hook/payment_return.tpl');
	}

	/**
	 * Check currency
	 */
    public function checkCurrency($cart)
	{
		$currency_order = new Currency((int)($cart->id_currency));
		$currencies_module = $this->getCurrency((int)$cart->id_currency);

		if (is_array($currencies_module))
			foreach ($currencies_module as $currency_module)
				if ($currency_order->id == $currency_module['id_currency'])
					return true;

		return false;
	}

	/**
	 * Render config page form
	 */
	public function renderForm()
	{
		$production_mode_options = array(
			array( 'id' => 'production_mode_on', 'value' => 1, 'label' => $this->l('Yes')),
			array( 'id' => 'production_mode_off', 'value' => 0, 'label' => $this->l('No')),
		  );

		$fields_form = array(
			'form' => array(
				'legend' => array(
					'title' => $this->l('Configuration'),
					'icon' => 'icon-cog'
				),
				'input' => array(
					array(
						'type' => 'text',
						'label' => $this->l('Merchant name'),
						'name' => 'merchant_name',
						'required' => true
					),
					array(
						'type' => 'text',
						'label' => $this->l('Merchant key'),
						'name' => 'merchant_key',
						'required' => true
					),
					array(
						'type' => 'text',
						'label' => $this->l('Authorization header'),
						'name' => 'auth_header',
						'required' => true
					),
					array(
						'type' => 'switch',
						'label' => $this->l('Enable Production Mode'),
						'name' => 'production_mode',
						'required' => false,
						'is_bool' => true,
						'values' => $production_mode_options
					)
				),
				'submit' => array(
					'title' => $this->l('Update settings'),
				)
			),
		);

		$helper = new HelperForm();
		$helper->show_toolbar = false;
		$helper->table = $this->table;
		$lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
		$helper->default_form_language = $lang->id;
		$helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
		$this->fields_form = array();
		$helper->id = (int)Tools::getValue('id_carrier');
		$helper->identifier = $this->identifier;
		$helper->submit_action = 'btnSubmit';
		$helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
		$helper->token = Tools::getAdminTokenLite('AdminModules');
		$helper->tpl_vars = array(
			'fields_value' => $this->getConfigFieldsValues(),
			'languages' => $this->context->controller->getLanguages(),
			'id_language' => $this->context->language->id
		);
		return $helper->generateForm(array($fields_form));
	}

	/**
	 * Gets config fields values
	 */

	public function getConfigFieldsValues()

	{
		return array(
			'merchant_name' => Tools::getValue('merchant_name', Configuration::get('AB_MERCHANT_NAME')),
			'merchant_key' => Tools::getValue('merchant_key', Configuration::get('AB_MERCHANT_KEY')),
			'auth_header' => Tools::getValue('auth_header', Configuration::get('AB_AUTH_HEADER')),
			'production_mode' => Tools::getValue('production_mode', Configuration::get('AB_PRODUCTION_MODE'))
		);
	}

	/**
	 * Order confirmation
	 */
	public function orangeMoneyOrderConfirmation($order, $customer, $state){

		$history = new OrderHistory();
        $history->id_order = $order->id;
        $history->changeIdOrderState($state,$order->id);
        Db::getInstance()->update('order_history',array('id_order_state' => $state),'id_order='.$order->id,1,false,true,true);

		if ($state == Configuration::get('PS_OS_PAYMENT')){
			Tools::redirect('index.php?controller=order-confirmation&id_cart='.(int)$order->id_cart.'&id_module='.(int)$this->id.'&id_order='.$order->id.'&key='.$customer->secure_key);
		} else {
			Tools::redirect('index.php');
		}
	}

	/**
	 * Write in Log file
	 */
	public function writeInLogFile($data,$level){

		$day = gmdate("Y-m-d");
		$logfile = _PS_MODULE_DIR_.'orangemoneycmr/logs/orangemoneycmr_'.$day.'.log';
		error_log("\r\n".'['.gmdate("Y-m-d H:i:s").'] '.$level.' '.$data, 3, $logfile);
	}
}
