<?php
/**
* 2007-2015 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    PrestaShop SA <contact@prestashop.com>
*  @copyright 2007-2015 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

require_once dirname(__FILE__).'/controllers/QuotationFormController.class.php';

if (!defined('_PS_VERSION_'))
    exit;

class quotation extends Module
{
    /** @var string Module js path (eg. '/shop/modules/module_name/js/') */
    protected $js_path = null;

    /** @var string Module css path (eg. '/shop/modules/module_name/css/') */
    protected $css_path = null;

    /** @var string Module css path (eg. '/shop/modules/module_name/css/') */
    protected $sql_path = null;

    /** @var protected string cache filled with informations */
    protected $cache_path;

    protected $config_form = false;
    protected static $lang_cache;

    private $subject = array();

    public function sendMail($template, $to, $id_lang, $template_vars = array())
    {
        $template_path = dirname(__FILE__).'/mails/';
        $iso_lang = Language::getIsoById($id_lang);

        if(!file_exists($template_path.'/'.$iso_lang.'/'.$template.'.html'))
            $this->createMailFolder($iso_lang);
        // TODO - Set default language if can't create files !!!!!!
        return Mail::Send($id_lang, $template, $this->subject[$template], $template_vars, $to, null, null, null, null, null, $template_path, false, Shop::getContextShopID());
    }

    private function copy_recursive($source, $dest)
    {
        if (is_dir($source))
        {
            $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
            );

            foreach ($iterator as $file)
            {
                if ($file->isDir())
                {
                    mkdir($dest.DIRECTORY_SEPARATOR.$iterator->getSubPathName());
                }
                else
                {
                    copy($file, $dest.DIRECTORY_SEPARATOR.$iterator->getSubPathName());
                }
            }
        }
        else
        {
            copy($source, $dest);
        }
    }

    private function createMailFolder($iso_lang)
    {
        $src = dirname(__FILE__).'/mails/en';
        $dir = opendir($src);
        $dst = dirname(__FILE__).'/mails/'.$iso_lang;
        @mkdir($dst);
        while(false !== ( $file = readdir($dir)) ) {
            if (( $file != '.' ) && ( $file != '..' )) {
                if ( is_dir($src . '/' . $file) ) {
                    $this->copy_recursive($src . '/' . $file, $dst . '/' . $file);
                }
                else {
                    copy($src . '/' . $file,$dst . '/' . $file);
                }
            }
        }
        closedir($dir);
    }

    public function __construct()
    {
        $this->name = 'quotation';
        $this->tab = 'billing_invoicing';
        $this->version = '1.0.27';
        $this->author = 'PrestaShop';
        $this->need_instance = 0;
        $this->module_key = '8aef8d7c3ee50ae34994eb73f755cf47';

        parent::__construct();
        /**
         * Set $this->bootstrap to true if your module is compliant with bootstrap (PrestaShop 1.6)
         */
        $this->bootstrap = true;
        $this->js_path = $this->_path.'js/';
        $this->css_path = $this->_path.'css/';
        $this->sql_path = dirname(__FILE__).'/sql/';
        $this->cache_path = $this->local_path.'cache/';

        $this->displayName = $this->l('Quotation');
        $this->description = $this->l('Allow your customers to ask for a quotation on your shop online!');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall ?');
        $this->subject['status_change'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Quotation status changed');
        $this->subject['status_change_client'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Quotation status changed');
        $this->subject['new_quotation'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('New quotation created');
        $this->subject['quotation_validated'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Your quotation has been validated');
        $this->subject['quotation_ordered'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Quotation ordered');
        $this->subject['quotation_updated'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Quotation modified');
        $this->subject['new_quotation_client'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('Quotation sended');
        $this->subject['new_message'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('New message');
        $this->subject['new_message_client'] = Configuration::get('PS_SHOP_NAME').' - '.$this->l('New message');
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        Configuration::updateValue('QUOTATION_LIVE_MODE', false);

        include(dirname(__FILE__).'/sql/install.php');

        $token = uniqid(rand(), true);
        Configuration::updateValue('QUOTATION_TOKEN', $token);
        unset($token);

        Configuration::updateGlobalValue('PS_CART_RULE_FEATURE_ACTIVE', '1');
        return parent::install() &&
            $this->registerHook('header') &&
            $this->registerHook('backOfficeHeader') &&
            $this->registerHook('displayProductButtons') &&
            $this->registerHook('displayTop') &&
            $this->registerHook('actionAuthentication') &&
            $this->registerHook('actionCustomerAccountAdd') &&
            $this->registerHook('displayCustomerAccount') &&
            $this->registerHook('orderConfirmation') &&
            $this->registerHook('actionObjectLanguageAddAfter') &&
            $this->installTab();
    }

    public function uninstall()
    {
        Configuration::deleteByName('QUOTATION_LIVE_MODE');

        include(dirname(__FILE__).'/sql/uninstall.php');

        return parent::uninstall() &&
            $this->unregisterHook('displayProductButtons') &&
            $this->unregisterHook('displayTop') &&
            $this->unregisterHook('actionAuthentication') &&
            $this->unregisterHook('displayCustomerAccount') &&
            $this->unregisterHook('header') &&
            $this->unregisterHook('orderConfirmation') &&
            $this->unregisterHook('backOfficeHeader');
    }

    private function installTab()
    {
        $tab = new Tab();
        $tab->active = 1;
        $tab->class_name = 'AdminQuotation';
        $tab->name = array();
        foreach (Language::getLanguages(true) as $lang)
            $tab->name[$lang['id_lang']] = 'Quotation';
        unset($lang);
        $tab->id_parent = -1;
        $tab->module = $this->name;
        return $tab->add();
    }

    /**
    * Uninstall Tab
    * @return boolean
    */
    private function uninstallTab()
    {
        $id_tab = (int)Tab::getIdFromClassName('AdminQuotation');
        if ($id_tab)
        {
            $tab = new Tab($id_tab);
            if (Validate::isLoadedObject($tab))
                return $tab->delete();
            else
                return false;
        }
        else
            return true;
    }

    /**
    * Loads asset resources
    */
    public function loadAsset()
    {
        $css_compatibility = $js_compatibility = array();

        // Load CSS
        $css = array(
            $this->css_path.'bootstrap-responsive.min.css',
            $this->css_path.'DT_bootstrap.css',
            $this->js_path.'ckeditor/content.css',
            $this->js_path.'jquery.contextMenu.css',
            $this->css_path.'sweetalert.css',
            $this->css_path.'jquery.dataTables.min.css',
            $this->css_path.'DT_bootstrap.css',
            $this->css_path.'faq.css',
            // $this->css_path.'font-awesome.min.css',
            $this->css_path.$this->name.'.css'
        );
        if (version_compare(_PS_VERSION_, '1.6', '<'))
        {
            $css_compatibility = array(
                $this->css_path.'bootstrap.min.css',
                $this->css_path.'bootstrap.extend.css',
                $this->css_path.'font-awesome.min.css',
            );
            $css = array_merge($css_compatibility, $css);
        }
        $this->context->controller->addCSS($css, 'all');

        // Load JS
        $jss = array(
            $this->js_path.'bootstrap-select.min.js',
            $this->js_path.'bootstrap-dialog.js',
            $this->js_path.'ckeditor/ckeditor.js',
            $this->js_path.'contextmenu/jquery.contextMenu.js',
            $this->js_path.'sweetalert.min.js',
            $this->js_path.'jquery.dataTables.min.js',
            $this->js_path.'DT_bootstrap.js',
            $this->js_path.'faq.js',
            $this->js_path.$this->name.'.js'
        );

        if (method_exists($this->context->controller, 'addJquery'))
            $this->context->controller->addJquery('2.1.0', $this->js_path);

        if (version_compare(_PS_VERSION_, '1.6', '<'))
        {
            $js_compatibility = array(
                $this->js_path.'bootstrap.min.js'
            );
            $jss = array_merge($jss, $js_compatibility);
        }
        $this->context->controller->addJS($jss);
        $this->context->controller->addJqueryUI('ui.draggable');
        $this->context->controller->addJqueryUI('ui.sortable');
        $this->context->controller->addJqueryUI('ui.position');
        // Clean memory
        unset($jss, $css, $js_compatibility, $css_compatibility);
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        //FAQ API CALL
        include_once('classes/APIFAQClass.php');
        $api = new APIFAQ();
        $api_json = Tools::jsonDecode($api->getData($this));
        $this->context->smarty->assign('apifaq', $api_json->categories);

        $this->handleForms();
        $this->initRight('forms');
        $this->initRight('mails');
        $this->loadAsset();
        if (version_compare(_PS_VERSION_, '1.6', '<'))
        {
            // Clean the code use tpl file for html
            $tab = '&tab_module='.$this->tab;
            $token_mod = '&token='.Tools::getAdminTokenLite('AdminModules');
            $token_pos = '&token='.Tools::getAdminTokenLite('AdminModulesPositions');
            $token_trad = '&token='.Tools::getAdminTokenLite('AdminTranslations');

            $this->context->smarty->assign(array(
                'module_active' => (bool)$this->active,
                'module_trad' => 'index.php?controller=AdminTranslations'.$token_trad.'&type=modules&lang=',
                'module_hook' => 'index.php?controller=AdminModulesPositions'.$token_pos.'&show_modules='.$this->id,
                'module_back' => 'index.php?controller=AdminModules'.$token_mod.$tab.'&module_name='.$this->name,
                'module_form' => 'index.php?controller=AdminModules&configure='.$this->name.$token_mod.$tab.'&module_name='.$this->name,
                'module_reset' => 'index.php?controller=AdminModules'.$token_mod.'&module_name='.$this->name.'&reset'.$tab
            ));
            // Clean memory
            unset($tab, $token_mod, $token_pos, $token_trad);
        }

        $ps_url = Tools::usingSecureMode() ? Tools::getShopDomainSsl(true) : Tools::getShopDomain(true);
        $ps_url .= __PS_BASE_URI__;

        $controller_name = 'AdminQuotation';
        $controller_url = $this->context->link->getAdminLink($controller_name);
        // $controller_url = $ps_url.'index.php?fc=module&module=quotation&controller=AdminQuotation';

        /* Language for documentation in back-office */
        $iso_code = Context::getContext()->language->iso_code;
        $lang = 'EN';
        if ($iso_code == 'fr' || $iso_code == 'FR')
            $lang = 'FR';

        $languages = Language::getLanguages(true, $this->context->shop->id);

        if(Tools::getIsset('quotation_tab'))
            $tab = Tools::getValue('quotation_tab');
        else
            $tab = 'doc';

        $formActive = QuotationFormController::getActiveForms();

        $this->context->smarty->assign(array(
            'module_name' => $this->name,
            'module_version' => $this->version,
            'debug_mode' => (int)_PS_MODE_DEV_,
            'lang_select' => self::$lang_cache,
            'current_id_tab' => (int)$this->context->controller->id,
            'controller_url' => $controller_url,
            'controller_name' => $controller_name,
            'module_display' => $this->displayName,
            'multishop' => (int)Shop::isFeatureActive(),
            'guide_link' => 'docs/readme_'.$lang.'.pdf',
            'ps_version' => (bool)version_compare(_PS_VERSION_, '1.6', '>'),
            'rewriting_allow' => (int)Configuration::get('PS_REWRITING_SETTINGS'),
            'id_shop' => $this->context->shop->id,
            'id_shop_group' => $this->context->shop->id_shop_group,
            'languages' => $languages,
            'lang_count' => count($languages),
            'lang_default' => Configuration::get('PS_LANG_DEFAULT', null, $this->context->shop->id_shop_group, $this->context->shop->id),
            'nb_quotation' => QuotationFormController::getMaxId(),
            'forms'	=>	QuotationFormController::getForms(),
            'quotations' => QuotationFormController::getQuotations(),
            'statuts' => QuotationFormController::getStatuts(),
            'quotation_live_mode' => Configuration::get('QUOTATION_LIVE_MODE'),
            'quotation_ip' => Configuration::get('QUOTATION_IP'),
            'quotation_prefix' => Configuration::get('QUOTATION_PREFIX'),
            'formActive' => $formActive,
            'tab'	=>	$tab,
            'isWritable' => (int)(is_writable('../modules/quotation/mails/') && is_writable('../modules/quotation/forms/')),
            'groups' => $this->getGroups()
        ));

        return $this->display(__FILE__, 'views/templates/admin/configuration.tpl');
    }

    private function handleForms()
    {
        if(!Tools::getIsset('submit'))
            return;

        if(Tools::getIsset('submit_conf') && Tools::getValue('submit_conf') == 1) {
            $this->handleConf();
        }

    }

    private function handleConf()
    {
        $return = true;

        $groups = Tools::getValue('groups');

        foreach($groups as $id_group => $active) {
            if(!Db::getInstance()->execute('REPLACE INTO '._DB_PREFIX_.'quot_group  VALUES ('.(int)$id_group.', '.(int)$active.')'))
                $return = false;
        }
        return $return;
    }

    private function getGroups()
    {
        $groups = Group::getGroups($this->context->language->id, $this->context->shop->id);
        $authorized_groups = $this->getAuthorizedGroups();

        foreach($groups as &$group) {
            if(!in_array($group['id_group'], $authorized_groups))
                $group['active'] = 0;
            else
                $group['active'] = 1;
        }
        return $groups;
    }
    private function getAuthorizedGroups()
    {
        $groups = array();
        foreach(Db::getInstance()->executeS('SELECT id_group FROM '._DB_PREFIX_.'quot_group WHERE active = 1') as $group) {
            $groups[] = $group['id_group'];
        }
        return $groups;
    }

    private function quotationForm()
    {
        if(!Tools::isSubmit('saveQuotation'))
            return true;

        Tools::safePostVars();
        $langs = array();
        foreach($_POST as $key => $vars)
        {
            if(!strstr($key, 'quotation_name_'))
                continue;
            $array = explode('quotation_name_', $key);
            $langs[$array[1]] = $val;
        }
        $id_quotation = (int)Tools::getValue('id_quotation');

        foreach($langs as $key => $lang)
        {
            $form = new Form($id_quotation, $key, $val);
            $form->save();
        }
    }

    private function quotationList()
    {

    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitQuotationModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            .'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(), /* Add values for your inputs */
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm()
    {
        return array(
            'form' => array(
                'legend' => array(
                'title' => $this->l('Settings'),
                'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Live mode'),
                        'name' => 'QUOTATION_LIVE_MODE',
                        'is_bool' => true,
                        'desc' => $this->l('Use this module in live mode'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => true,
                                'label' => $this->l('Enabled')
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => false,
                                'label' => $this->l('Disabled')
                            )
                        ),
                    ),
                    array(
                        'col' => 3,
                        'type' => 'text',
                        'prefix' => '<i class="icon icon-envelope"></i>',
                        'desc' => $this->l('Enter a valid email address'),
                        'name' => 'QUOTATION_ACCOUNT_EMAIL',
                        'label' => $this->l('Email'),
                    ),
                    array(
                        'type' => 'password',
                        'name' => 'QUOTATION_ACCOUNT_PASSWORD',
                        'label' => $this->l('Password'),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return array(
            'QUOTATION_LIVE_MODE' => Configuration::get('QUOTATION_LIVE_MODE', true),
            'QUOTATION_ACCOUNT_EMAIL' => Configuration::get('QUOTATION_ACCOUNT_EMAIL', 'contact@prestashop.com'),
            'QUOTATION_ACCOUNT_PASSWORD' => Configuration::get('QUOTATION_ACCOUNT_PASSWORD', null),
        );
    }

    /**
    * Get Language
    * @return array Lang
    */
    private function getLang()
    {
        self::$lang_cache = TinyCache::getCache('language');

        if (self::$lang_cache === null || empty(self::$lang_cache))
        {
            if ($languages = Language::getLanguages())
            {
                foreach ($languages as &$row)
                {
                    $exprow = explode(' (', $row['name']);
                    $subtitle = (isset($exprow[1]) ? trim(Tools::substr($exprow[1], 0, -1)) : '');
                    self::$lang_cache[$row['iso_code']] = array (
                        'id' => (int)$row['id_lang'],
                        'title' => trim($exprow[0]),
                        'subtitle' => $subtitle
                    );
                }
                // Cache Data
                TinyCache::setCache('language', self::$lang_cache);
                // Clean memory
                unset($row, $exprow, $subtitle, $languages);
            }
        }
    }

    /**
     * Save form data.
     */
    protected function _postProcess()
    {
        $form_values = $this->getConfigFormValues();

        foreach (array_keys($form_values) as $key)
            Configuration::updateValue($key, Tools::getValue($key));
    }

    /**
    * Add the CSS & JavaScript files you want to be loaded in the BO.
    */
    public function hookBackOfficeHeader()
    {
        $module = trim(Tools::getValue('module_name'));
        $configure = trim(Tools::getValue('configure'));
        $controller_name = pSQL(trim(Tools::getValue('controller')));

        if ($controller_name === 'AdminModules' && ($module === $this->name || $configure === $this->name)) {
            $this->context->controller->addJS($this->_path.'js/back.js');
            $this->context->controller->addCSS($this->_path.'css/back.css');
            $this->context->controller->addCSS($this->_path.'css/quotatoin_messages.css');
        }
    }

    /**
     * Add the CSS & JavaScript files you want to be added on the FO.
     */
    public function hookHeader()
    {
        if(!$this->isLive())
            return;
        $this->context->controller->addJS($this->_path.'/js/front.js');
        $this->context->controller->addCSS($this->_path.'/css/front.css');
        $this->context->controller->addCSS($this->_path.'/css/quotationblock.css');
        $this->context->controller->addCSS($this->_path.'css/quotatoin_messages.css');
    }

    public function hookDisplayProductButtons($args)
    {
        if(!$this->isLive())
            return;
        $id_shop 		= Context::getContext()->cart->id_shop;
        $id_customer 	= Context::getContext()->cart->id_customer;
        $token 			= Configuration::get('QUOTATION_TOKEN');

        $currency_sign = $this->context->currency->sign;

        $ps_url = Tools::usingSecureMode() ? Tools::getShopDomainSsl(true) : Tools::getShopDomain(true);
        $ps_url .= __PS_BASE_URI__;

        $quotation_url = 'index.php?fc=module&module=quotation&controller=quotation';
        $quotation_url_customerQuotations = $ps_url.'?fc=module&module=quotation&controller=customerquotations';

        if(isset($this->context->cookie->total_cart))
            $total_cart = number_format($this->context->cookie->total_cart+$args['product']->getPrice(), 2);
        else
            $total_cart = 0;

        if(isset($this->context->cookie->total_product))
            $total_product = $this->context->cookie->total_product+1;
        else
            $total_product = 0;

        $this->context->smarty->assign(array(
            'ajax_path' => $ps_url.'modules'.'/'.$this->name.'/ajax.php',
            'id_shop' => $id_shop,
            'id_customer' => $id_customer,
            'quotation_token' => $token,
            'product' => $args['product'],
            'currency_sign' => $currency_sign,
            'quotation_url' => $quotation_url,
            'quotation_url_customerQuotations' => $quotation_url_customerQuotations,
            'total_cart' => $total_cart,
            'total_product' => $total_product,
            'id_product' => Tools::getValue('id_product'),
            // 'link' => $test,
            'PS_CATALOG_MODE' => Configuration::get('PS_CATALOG_MODE')
        ));
        return $this->display(__FILE__, 'views/templates/hook/hookDisplayProductButtons.tpl');
    }

    public function hookDisplayTop()
    {
        if(!$this->isLive())
            return;
        $this->initQuotation();
        $currency_sign = $this->context->currency->sign;

        if(isset($this->context->cookie->products))
        {
            $product_list = Tools::jsonDecode($this->context->cookie->products, true);
            // d($product_list);
            $products = array();
            $i = 0;

            foreach ($product_list as $key => $value) {
                $product_name = explode(':', Product::getProductName($value['id_product'], $value['id_combination']));

                $products[$i]['id_product'] = $value['id_product'];
                $products[$i]['id_combination'] = $value['id_combination'];
                $products[$i]['quantity'] = $value['quantity'];
                $products[$i]['date_add'] = $value['date_add'];
                $products[$i]['price'] = number_format($value['price'], 2);
                $products[$i]['img'] = $value['img'];
                $products[$i]['link'] = $value['link'];
                $products[$i]['name'] = $product_name[0];
                $products[$i]['combination_name'] = $product_name[1];
                $products[$i]['reference'] = $this->getRefById($value['id_product'], $value['id_combination']);
                $products[$i]['unit_price'] = number_format(Product::getPriceStatic($value['id_product'], $usetax = true, $value['id_combination']), 2);
                $i++;
            }
            $has_product = true;
        }
        else
        {
            $products = array();
            $has_product = false;
        }

        if(isset($this->context->cookie->id_quotation))
            $id_quotation = $this->context->cookie->id_quotation;
        else
            $id_quotation = 0;

        if(isset($this->context->cookie->total_cart))
            $total_cart = number_format($this->context->cookie->total_cart, 2);
        else
            $total_cart = 0;

        if(isset($this->context->cookie->total_product))
            $total_product = $this->context->cookie->total_product;
        else
            $total_product = 0;

        $ps_url = Tools::usingSecureMode() ? Tools::getShopDomainSsl(true) : Tools::getShopDomain(true);
        $ps_url .= __PS_BASE_URI__;

        $quotation_url = $ps_url.'index.php?fc=module&module=quotation&controller=quotation';
        $quotation_url_customerQuotations = $ps_url.'index.php?fc=module&module=quotation&controller=customerquotations';

        $this->context->controller->addJS($this->js_path.'sweetalert.min.js');

        $this->context->controller->addCSS($this->css_path.'sweetalert.css');

        if(Group::getPriceDisplayMethod(Context::getContext()->customer->id_default_group) == 1)
            $tax_text = $this->l('Without taxes');
        else
            $tax_text = $this->l('With taxes');

        $this->context->smarty->assign(array(
            'id_quotation' => $id_quotation,
            'quotation_products' => $products,
            'quotation_has_product' => $has_product,
            'currency_sign' => $currency_sign,
            'total_cart' => $total_cart,
            'total_product' => $total_product,
            'quotation_url' => $quotation_url,
            'quotation_url_customerQuotations' => $quotation_url_customerQuotations,
            'tax_text' => $tax_text,
            'PS_CATALOG_MODE' => Configuration::get('PS_CATALOG_MODE')
            )
        );
        return $this->display(__FILE__, 'views/templates/hook/hookDisplayTop.tpl');
    }

    private function getRefById($id_product, $id_combination)
    {
        $sql = 'SELECT pa.reference FROM `'._DB_PREFIX_.'product_attribute` pa
                WHERE id_product='.(int)$id_combination.'';
        $result = Db::getInstance()->getRow($sql);

        if (!empty($result)) {
            $sql = 'SELECT p.reference FROM `'._DB_PREFIX_.'product` p
                    WHERE id_product='.(int)$id_product.'';
            $result = Db::getInstance()->getRow($sql);
        }

        $result = $result['reference'];

        return $result;
    }

    public function refreshCart()
    {
        if(isset($this->context->cookie->total_cart))
            $total_cart = number_format($this->context->cookie->total_cart, 2);
        else
            $total_cart = 0;

        return($total_cart);
    }

    private function initQuotation()
    {
        if(isset($this->context->cookie->id_quotation))
            $id_quotation = $this->context->cookie->id_quotation;
        else
            $id_quotation = null;

        $is_guest = !(int)$this->context->customer->id > 0;

        if(!$is_guest){
            $id_guest = null;
            $id_customer = (int)$this->context->customer->id;
            $filter_customer = true;
            $filter_guest = false;
        }
        else
        {
            $id_guest = (int)$this->context->customer->id_guest;
            $id_customer = null;
            $filter_customer = false;
            $filter_guest = true;
        }

        require_once dirname(__FILE__).'/classes/QuotationObj.class.php';

        $quotation = new QuotationObj($id_quotation, $id_customer, $id_guest, Context::getContext()->cart->id_shop);
        $quotation->init();
    }

    public function hookActionAuthentication()
    {
        if(!$this->isLive())
            return;
        $query = 'UPDATE '._DB_PREFIX_.'quot SET id_customer = '.(int)Context::getContext()->customer->id.' WHERE id_guest = '.(int)Context::getContext()->cart->id_guest ;
        return Db::getInstance()->execute($query);
    }

    public function hookActionCustomerAccountAdd($params)
    {
        if(!$this->isLive())
            return;
        $query = 'UPDATE '._DB_PREFIX_.'quot SET id_customer = '.(int)Context::getContext()->customer->id.' WHERE id_guest = '.(int)Context::getContext()->cart->id_guest ;
        return Db::getInstance()->execute($query);
    }
    public function hookDisplayCustomerAccount($params)
    {
        if(!$this->isLive())
            return;

        $ps_url = Tools::usingSecureMode() ? Tools::getShopDomainSsl(true) : Tools::getShopDomain(true);
        $this->context->smarty->assign('url', $ps_url.__PS_BASE_URI__.'index.php?fc=module&module=quotation&controller=customerquotations');

        return $this->display(__FILE__, 'views/templates/hook/hookDisplayCustomerAccount.tpl');
    }

    public function hookOrderConfirmation($params)
    {
        if(!$this->isLive())
            return;
        require_once dirname(__FILE__).'/controllers/QuotationFormController.class.php';
        $id_cart = Order::getCartIdStatic($params['objOrder']->id);
        QuotationFormController::setStatutByCart($id_cart, 6, false);
        $args = array();
        $args['{id_quot}'] = Configuration::get('QUOTATION_PREFIX').''.QuotationObj::getIdQuotByIdCart($id_cart);
        $args['{logo}'] =  _PS_BASE_URL_.__PS_BASE_URI__.'/img/'.Configuration::get('PS_LOGO', null, Context::getContext()->shop->id_shop_group, Context::getContext()->shop->id);
        $args['{customer_details}'] = Context::getContext()->cookie->customer_firstname.' '.Context::getContext()->cookie->customer_lastname.' (ID '.Context::getContext()->cookie->id_customer.')';
        $args['{shop_name}'] = Configuration::get('PS_SHOP_NAME', null, Shop::getContextShopGroupID(), Shop::getContextShopID());
        $args['{shopping_cart_ID}'] = $id_cart;

        $this->sendMail('quotation_ordered', Configuration::get('PS_SHOP_EMAIL'), Configuration::get('PS_LANG_DEFAULT', null, Shop::getContextShopGroupID(), Shop::getContextShopID()), $args);
    }

    public function hookactionObjectLanguageAddAfter($params)
    {
        if (!$this->active)
            return;

        $id_lang_default = Configuration::get('PS_LANG_DEFAULT', null, $this->context->shop->id_shop_group, $this->context->shop->id);
        $id_lang = $params['object']->id;

        $query = 'SELECT id_form, id_lang FROM '._DB_PREFIX_.'quot_form WHERE id_lang = '.(int)$id_lang_default;
        $results = Db::getInstance()->executeS($query);

        $query = 'INSERT INTO '._DB_PREFIX_.'quot_form SELECT id_form, '.(int)$id_lang.', form_name, active, id_shop, id_shop_group FROM '._DB_PREFIX_.'quot_form WHERE id_lang = '.(int)$id_lang_default;
        Db::getInstance()->execute($query);

        $query = 'INSERT INTO '._DB_PREFIX_.'quot_form_field SELECT id, id_form, label, name, type_field, value, id_parent, pos, '.(int)$id_lang.' FROM '._DB_PREFIX_.'quot_form_field WHERE id_lang = '.(int)$id_lang_default;
        Db::getInstance()->execute($query);

        foreach($results as $result)
        {
            copy(dirname(__FILE__).'/forms/'.$result['id_form'].'_'.$result['id_lang'].'.tpl', dirname(__FILE__).'/forms/'.$result['id_form'].'_'.$id_lang.'.tpl');
        }
    }

    public function isLive()
    {
        $return = true;

        $live_mode = Configuration::get('QUOTATION_LIVE_MODE');
        $quotation_ip = Configuration::get('QUOTATION_IP');
        $formActive = QuotationFormController::getActiveForms();

        if ((!$live_mode && $quotation_ip != Tools::getRemoteAddr()) || !in_array(Context::getContext()->customer->id_default_group, $this->getAuthorizedGroups())) {
            $return = false;
        }
        if (empty($formActive)) {
            $return = false;
        }
        return $return;
    }

    public static function initRight($path, $right = '0777'){
        $prems = Tools::substr(sprintf('%o', fileperms(dirname(__FILE__).'/'.$path)), -4);
        if ($prems != $right)
            @chmod(dirname(__FILE__).'/'.$path, octdec((int)$right));
    }
}
